@extends('layouts.admin')

@section('title', 'Edit Package')

@section('content')
<div class="dashboard-container">
    <div class="dashboard-card dashboard-card-full">
        <div class="dashboard-card-header">
            <h3>Edit Package</h3>
        </div>
        <div class="dashboard-card-body">
            <form action="{{ route('admin.packages.update', $package) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="name">Package Name</label>
                            <input type="text" name="name" id="name" class="form-control @error('name') has-error @enderror" value="{{ old('name', $package->name) }}" required>
                            @error('name')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description</label>
                            <textarea name="description" id="description" class="form-control @error('description') has-error @enderror" rows="4" required>{{ old('description', $package->description) }}</textarea>
                            @error('description')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="price">Price ({{ currency_symbol(config_value('store_currency', 'GBP')) }})</label>
                                    <input type="number" name="price" id="price" class="form-control @error('price') has-error @enderror" value="{{ old('price', $package->price) }}" step="0.01" min="0" required>
                                    @error('price')
                                        <div class="error-message">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="type">Package Type</label>
                                    <select name="type" id="type" class="form-control @error('type') has-error @enderror" required>
                                        <option value="">Select Type</option>
                                        @foreach($types as $type)
                                            <option value="{{ $type }}" {{ old('type', $package->type) == $type ? 'selected' : '' }}>{{ ucfirst(str_replace('_', ' ', $type)) }}</option>
                                        @endforeach
                                    </select>
                                    @error('type')
                                        <div class="error-message">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Features</label>
                            <div class="feature-list" id="feature-list">
                                @if(old('features'))
                                    @foreach(old('features') as $key => $feature)
                                        <div class="feature-item">
                                            <input type="text" name="features[]" class="form-control" value="{{ $feature }}">
                                            <button type="button" class="btn btn-sm btn-danger remove-feature">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        </div>
                                    @endforeach
                                @elseif($package->features)
                                    @foreach($package->features as $feature)
                                        <div class="feature-item">
                                            <input type="text" name="features[]" class="form-control" value="{{ $feature }}">
                                            <button type="button" class="btn btn-sm btn-danger remove-feature">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        </div>
                                    @endforeach
                                @else
                                    <div class="feature-item">
                                        <input type="text" name="features[]" class="form-control" placeholder="Enter a feature">
                                        <button type="button" class="btn btn-sm btn-danger remove-feature">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                @endif
                            </div>
                            <button type="button" class="btn btn-sm btn-secondary mt-2" id="add-feature">
                                <i class="fas fa-plus"></i> Add Feature
                            </button>
                        </div>
                        
                        <!-- Actions Section -->
                        <div class="form-group">
                            <label>Package Actions</label>
                            <div class="actions-list" id="actions-list">
                                @if(old('actions_type'))
                                    @foreach(old('actions_type') as $key => $actionType)
                                        <div class="action-item">
                                            <div class="row">
                                                <div class="col-md-3">
                                                    <select name="actions_type[]" class="form-control action-type-select">
                                                        <option value="">Select Action</option>
                                                        @foreach($actionTypes as $type)
                                                            <option value="{{ $type }}" {{ $actionType == $type ? 'selected' : '' }}>{{ ucfirst(str_replace('_', ' ', $type)) }}</option>
                                                        @endforeach
                                                    </select>
                                                </div>
                                                <div class="col-md-5">
                                                    <input type="text" name="actions_value[]" class="form-control" value="{{ old('actions_value')[$key] ?? '' }}" placeholder="Action value">
                                                </div>
                                                <div class="col-md-3 duration-field" style="{{ $actionType == 'rank' ? '' : 'display: none;' }}">
                                                    <input type="number" name="actions_duration[]" class="form-control" value="{{ old('actions_duration')[$key] ?? '0' }}" placeholder="Duration (days)" min="0">
                                                    <small class="text-muted">0 = Permanent</small>
                                                </div>
                                                <div class="col-md-1">
                                                    <button type="button" class="btn btn-sm btn-danger remove-action">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    @endforeach
                                @elseif($package->actions)
                                    @foreach($package->actions as $action)
                                        <div class="action-item">
                                            <div class="row">
                                                <div class="col-md-3">
                                                    <select name="actions_type[]" class="form-control action-type-select">
                                                        <option value="">Select Action</option>
                                                        @foreach($actionTypes as $type)
                                                            <option value="{{ $type }}" {{ $action['type'] == $type ? 'selected' : '' }}>{{ ucfirst(str_replace('_', ' ', $type)) }}</option>
                                                        @endforeach
                                                    </select>
                                                </div>
                                                <div class="col-md-5">
                                                    <input type="text" name="actions_value[]" class="form-control" value="{{ $action['value'] }}" placeholder="Action value">
                                                </div>
                                                <div class="col-md-3 duration-field" style="{{ $action['type'] == 'rank' ? '' : 'display: none;' }}">
                                                    <input type="number" name="actions_duration[]" class="form-control" value="{{ $action['duration'] ?? '0' }}" placeholder="Duration (days)" min="0">
                                                    <small class="text-muted">0 = Permanent</small>
                                                </div>
                                                <div class="col-md-1">
                                                    <button type="button" class="btn btn-sm btn-danger remove-action">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    @endforeach
                                @else
                                    <div class="action-item">
                                        <div class="row">
                                            <div class="col-md-3">
                                                <select name="actions_type[]" class="form-control action-type-select">
                                                    <option value="">Select Action</option>
                                                    @foreach($actionTypes as $type)
                                                        <option value="{{ $type }}">{{ ucfirst(str_replace('_', ' ', $type)) }}</option>
                                                    @endforeach
                                                </select>
                                            </div>
                                            <div class="col-md-5">
                                                <input type="text" name="actions_value[]" class="form-control" placeholder="Action value">
                                            </div>
                                            <div class="col-md-3 duration-field" style="display: none;">
                                                <input type="number" name="actions_duration[]" class="form-control" value="0" placeholder="Duration (days)" min="0">
                                                <small class="text-muted">0 = Permanent</small>
                                            </div>
                                            <div class="col-md-1">
                                                <button type="button" class="btn btn-sm btn-danger remove-action">
                                                    <i class="fas fa-times"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                @endif
                            </div>
                            <button type="button" class="btn btn-sm btn-secondary mt-2" id="add-action">
                                <i class="fas fa-plus"></i> Add Action
                            </button>
                            <small class="form-text text-muted mt-2">
                                <strong>Action Types:</strong><br>
                                - <strong>rank:</strong> Sets user's rank (e.g. "vip", "admin", "moderator") with optional duration<br>
                                - <strong>money:</strong> Gives in-game currency (e.g. "1000", "50000")<br>
                                - <strong>custom_lua:</strong> Executes custom Lua code on the server<br>
                                - <strong>console_command:</strong> Executes a console command (e.g. "ulx adduser {user_name} vip")<br><br>
                                
                                <strong>Available Placeholders:</strong><br>
                                <code>{user_name}</code> - Player's name<br>
                                <code>{user_steamid}</code> - Player's Steam ID<br>
                                <code>{package_name}</code> - Package name<br>
                                <code>{package_price}</code> - Final price paid<br>
                                <code>{original_price}</code> - Original price before discounts<br>
                                <code>{discount_amount}</code> - Discount amount<br>
                                <code>{transaction_id}</code> - Transaction ID<br>
                                <code>{current_date}</code> - Current date (Y-m-d)<br>
                                <code>{timestamp}</code> - Unix timestamp
                            </small>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="form-group">
                            <label for="image">Package Image</label>
                            <div class="image-preview-container">
                                <div class="image-preview" id="imagePreview">
                                    @if($package->image_path)
                                    <img src="{{ asset('storage/' . $package->image_path) }}" class="preview-img">
                                    @else
                                        <div class="no-image-text">No Image Selected</div>
                                    @endif
                                </div>
                                <input type="file" name="image" id="image" class="form-control-file @error('image') has-error @enderror" accept="image/*">
                            </div>
                            @error('image')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">Recommended size: 400x300px. Max file size: 2MB.</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="sort_order">Display Order</label>
                            <input type="number" name="sort_order" id="sort_order" class="form-control @error('sort_order') has-error @enderror" value="{{ old('sort_order', $package->sort_order) }}" min="0" required>
                            @error('sort_order')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="form-group">
                            <label class="section-label">
                                <i class="fas fa-boxes"></i> Stock Management
                            </label>
                            
                            <div class="form-check mb-3">
                                <input type="checkbox" name="stock_enabled" id="stock_enabled" value="1" 
                                    {{ old('stock_enabled', $package->stock_enabled ?? false) ? 'checked' : '' }}>
                                <label for="stock_enabled">Enable Stock Tracking</label>
                                <small class="form-text text-muted d-block">Track inventory and limit sales to a specific quantity</small>
                            </div>
                            
                            <div id="stock_fields" style="{{ old('stock_enabled', $package->stock_enabled ?? false) ? '' : 'display: none;' }}">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="stock_quantity">Total Stock</label>
                                            <input type="number" 
                                                name="stock_quantity" 
                                                id="stock_quantity" 
                                                class="form-control @error('stock_quantity') has-error @enderror" 
                                                value="{{ old('stock_quantity', $package->stock_quantity ?? '') }}" 
                                                min="1" 
                                                placeholder="100">
                                            @error('stock_quantity')
                                                <div class="error-message">{{ $message }}</div>
                                            @enderror
                                            <small class="form-text text-muted">Total available quantity</small>
                                        </div>
                                    </div>
                                    
                                    @if(isset($package) && $package->stock_enabled)
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="stock_remaining">Remaining Stock</label>
                                            <input type="number" 
                                                name="stock_remaining" 
                                                id="stock_remaining" 
                                                class="form-control @error('stock_remaining') has-error @enderror" 
                                                value="{{ old('stock_remaining', $package->stock_remaining ?? '') }}" 
                                                min="0" 
                                                max="{{ $package->stock_quantity ?? 999999 }}">
                                            @error('stock_remaining')
                                                <div class="error-message">{{ $message }}</div>
                                            @enderror
                                            <small class="form-text text-muted">Current available stock</small>
                                        </div>
                                    </div>
                                    @endif
                                </div>
                                
                                @if(isset($package) && $package->stock_enabled)
                                <div class="stock-info-box">
                                    <div class="stock-stat">
                                        <span class="stat-label">Sold:</span>
                                        <span class="stat-value">{{ ($package->stock_quantity - $package->stock_remaining) }}</span>
                                    </div>
                                    <div class="stock-stat">
                                        <span class="stat-label">Percentage:</span>
                                        <span class="stat-value">{{ $package->stock_percentage }}%</span>
                                    </div>
                                    <div class="stock-stat">
                                        <span class="stat-label">Status:</span>
                                        <span class="stock-badge {{ $package->stock_badge_class }}">
                                            @if($package->isSoldOut())
                                                Sold Out
                                            @elseif($package->stock_percentage <= 25)
                                                Low Stock
                                            @else
                                                In Stock
                                            @endif
                                        </span>
                                    </div>
                                </div>
                                @endif
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="section-label">
                                <i class="fas fa-calendar-alt"></i> Subscription Settings
                            </label>
                            
                            <div class="form-check mb-3">
                                <input type="checkbox" name="is_subscription" id="is_subscription" value="1" 
                                    {{ old('is_subscription', $package->is_subscription ?? false) ? 'checked' : '' }}>
                                <label for="is_subscription">This is a Subscription Package</label>
                                <small class="form-text text-muted d-block">Package access will expire after a set duration</small>
                            </div>
                            
                            <div id="subscription_fields" style="{{ old('is_subscription', $package->is_subscription ?? false) ? '' : 'display: none;' }}">
                                <div class="form-group">
                                    <label for="subscription_duration">Subscription Duration (Days)</label>
                                    <input type="number" 
                                        name="subscription_duration" 
                                        id="subscription_duration" 
                                        class="form-control @error('subscription_duration') has-error @enderror" 
                                        value="{{ old('subscription_duration', $package->subscription_duration ?? '') }}" 
                                        min="1" 
                                        placeholder="30">
                                    @error('subscription_duration')
                                        <div class="error-message">{{ $message }}</div>
                                    @enderror
                                    <small class="form-text text-muted">
                                        Common durations: 7 (1 week), 30 (1 month), 90 (3 months), 365 (1 year)
                                    </small>
                                </div>
                                
                                @if(isset($package) && $package->is_subscription && $package->subscription_duration)
                                <div class="subscription-preview">
                                    <i class="fas fa-info-circle"></i>
                                    Users will have access for <strong>{{ $package->subscription_duration_human }}</strong> after purchase
                                </div>
                                @endif
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Assign to Servers</label>
                            <div class="server-checkboxes">
                                @foreach($servers as $server)
                                    <div class="form-check">
                                        <input type="checkbox" name="servers[]" id="server_{{ $server->id }}" value="{{ $server->id }}" 
                                               {{ in_array($server->id, old('servers', $assignedServers)) ? 'checked' : '' }}>
                                        <label for="server_{{ $server->id }}">{{ $server->name }}</label>
                                    </div>
                                @endforeach
                                @if($servers->isEmpty())
                                    <p class="text-muted">No servers available. <a href="{{ route('admin.servers.create') }}">Create a server</a> first.</p>
                                @endif
                            </div>
                            @error('servers')
                                <div class="error-message">{{ $message }}</div>
                            @enderror
                            <small class="form-text text-muted">Select which servers this package should be available on. Leave unchecked to make it available on all servers.</small>
                        </div>
                        
                        <div class="form-check">
                            <input type="checkbox" name="is_active" id="is_active" value="1" {{ old('is_active', $package->is_active) ? 'checked' : '' }}>
                            <label for="is_active">Active</label>
                        </div>
                    </div>
                </div>
                
                <div class="form-group mt-4">
                    <button type="submit" class="btn btn-primary">Update Package</button>
                    <a href="{{ route('admin.packages.index') }}" class="btn btn-secondary">Cancel</a>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Stock management toggle
        const stockEnabled = document.getElementById('stock_enabled');
        const stockFields = document.getElementById('stock_fields');
        
        if (stockEnabled) {
            stockEnabled.addEventListener('change', function() {
                stockFields.style.display = this.checked ? 'block' : 'none';
                
                // Clear values when disabled
                if (!this.checked) {
                    document.getElementById('stock_quantity').value = '';
                    const stockRemaining = document.getElementById('stock_remaining');
                    if (stockRemaining) stockRemaining.value = '';
                }
            });
        }
        
        // Subscription toggle
        const isSubscription = document.getElementById('is_subscription');
        const subscriptionFields = document.getElementById('subscription_fields');
        
        if (isSubscription) {
            isSubscription.addEventListener('change', function() {
                subscriptionFields.style.display = this.checked ? 'block' : 'none';
                
                // Clear value when disabled
                if (!this.checked) {
                    document.getElementById('subscription_duration').value = '';
                }
            });
        }

        // Image preview
        const imageInput = document.getElementById('image');
        const imagePreview = document.getElementById('imagePreview');
        
        imageInput.addEventListener('change', function() {
            if (this.files && this.files[0]) {
                const reader = new FileReader();
                
                reader.onload = function(e) {
                    imagePreview.innerHTML = `<img src="${e.target.result}" class="preview-img">`;
                }
                
                reader.readAsDataURL(this.files[0]);
            } else {
                imagePreview.innerHTML = `<div class="no-image-text">No Image Selected</div>`;
            }
        });
        
        // Features
        const addFeatureBtn = document.getElementById('add-feature');
        const featureList = document.getElementById('feature-list');
        
        addFeatureBtn.addEventListener('click', function() {
            const featureItem = document.createElement('div');
            featureItem.className = 'feature-item';
            featureItem.innerHTML = `
                <input type="text" name="features[]" class="form-control" placeholder="Enter a feature">
                <button type="button" class="btn btn-sm btn-danger remove-feature">
                    <i class="fas fa-times"></i>
                </button>
            `;
            featureList.appendChild(featureItem);
            
            // Add event listener to the new remove button
            featureItem.querySelector('.remove-feature').addEventListener('click', function() {
                featureList.removeChild(featureItem);
            });
        });
        
        // Add event listeners to existing remove buttons
        document.querySelectorAll('.remove-feature').forEach(button => {
            button.addEventListener('click', function() {
                this.closest('.feature-item').remove();
            });
        });
        
        // Actions
        const addActionBtn = document.getElementById('add-action');
        const actionsList = document.getElementById('actions-list');
        
        // Function to toggle duration field visibility
        function toggleDurationField(actionItem) {
            const typeSelect = actionItem.querySelector('.action-type-select');
            const durationField = actionItem.querySelector('.duration-field');
            
            if (typeSelect.value === 'rank') {
                durationField.style.display = '';
            } else {
                durationField.style.display = 'none';
            }
        }
        
        // Add event listeners to existing action type selects
        document.querySelectorAll('.action-type-select').forEach(select => {
            select.addEventListener('change', function() {
                toggleDurationField(this.closest('.action-item'));
            });
        });
        
        addActionBtn.addEventListener('click', function() {
            const actionItem = document.createElement('div');
            actionItem.className = 'action-item';
            actionItem.innerHTML = `
                <div class="row">
                    <div class="col-md-3">
                        <select name="actions_type[]" class="form-control action-type-select">
                            <option value="">Select Action</option>
                            <option value="rank">Rank</option>
                            <option value="money">Money</option>
                            <option value="custom_lua">Custom Lua</option>
                            <option value="console_command">Console Command</option>
                        </select>
                    </div>
                    <div class="col-md-5">
                        <input type="text" name="actions_value[]" class="form-control" placeholder="Action value">
                    </div>
                    <div class="col-md-3 duration-field" style="display: none;">
                        <input type="number" name="actions_duration[]" class="form-control" value="0" placeholder="Duration (days)" min="0">
                        <small class="text-muted">0 = Permanent</small>
                    </div>
                    <div class="col-md-1">
                        <button type="button" class="btn btn-sm btn-danger remove-action">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            `;
            actionsList.appendChild(actionItem);
            
            // Add event listener to the new type select
            const newTypeSelect = actionItem.querySelector('.action-type-select');
            newTypeSelect.addEventListener('change', function() {
                toggleDurationField(actionItem);
            });
            
            // Add event listener to the new remove button
            actionItem.querySelector('.remove-action').addEventListener('click', function() {
                actionsList.removeChild(actionItem);
            });
        });
        
        // Add event listeners to existing remove action buttons
        document.querySelectorAll('.remove-action').forEach(button => {
            button.addEventListener('click', function() {
                this.closest('.action-item').remove();
            });
        });
    });
</script>
@endpush


@push('styles')
<style>
    .section-label {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 600;
        font-size: 1rem;
        margin-bottom: 12px;
        color: var(--text-primary);
    }

    .stock-info-box {
        background: rgba(var(--primary-rgb), 0.1);
        border-radius: var(--border-radius-md);
        padding: 15px;
        margin-top: 15px;
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 15px;
    }

    .stock-stat {
        display: flex;
        flex-direction: column;
        align-items: center;
        text-align: center;
    }

    .stat-label {
        font-size: 0.85rem;
        color: var(--light-gray);
        margin-bottom: 5px;
    }

    .stat-value {
        font-size: 1.2rem;
        font-weight: 600;
        color: var(--text-primary);
    }

    .stock-badge {
        padding: 4px 12px;
        border-radius: var(--border-radius-sm);
        font-size: 0.85rem;
        font-weight: 600;
    }

    .stock-badge.in-stock {
        background: rgba(40, 167, 69, 0.2);
        color: #28a745;
    }

    .stock-badge.low-stock {
        background: rgba(255, 193, 7, 0.2);
        color: #ffc107;
    }

    .stock-badge.sold-out {
        background: rgba(220, 53, 69, 0.2);
        color: #dc3545;
    }

    .subscription-preview {
        background: rgba(var(--primary-rgb), 0.1);
        border-left: 3px solid var(--primary);
        padding: 12px 15px;
        border-radius: var(--border-radius-sm);
        margin-top: 10px;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .subscription-preview i {
        color: var(--primary);
        font-size: 1.1rem;
    }
    .image-preview-container {
        margin-bottom: 15px;
    }
    .image-preview {
        width: 100%;
        height: 200px;
        border: 1px dashed #ced4da;
        border-radius: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        overflow: hidden;
        margin-bottom: 10px;
    }
    .preview-img {
        max-width: 100%;
        max-height: 100%;
        object-fit: contain;
    }
    .no-image-text {
        color: #6c757d;
    }
    .feature-item, .action-item {
        display: flex;
        align-items: center;
        margin-bottom: 10px;
    }
    .feature-item .form-control {
        flex-grow: 1;
        margin-right: 10px;
    }
    .feature-item .btn {
        flex-shrink: 0;
    }
    .action-item {
        margin-bottom: 15px;
    }
    .server-checkboxes {
        max-height: 200px;
        overflow-y: auto;
        border: 1px solid #606060;
        border-radius: 4px;
        padding: 10px;
        background-color: #2a2a2a;
    }
    .server-checkboxes .form-check {
        margin-bottom: 8px;
    }
    .server-checkboxes .form-check:last-child {
        margin-bottom: 0;
    }
</style>
@endpush